<?php
header('Content-Type: application/json');

require_once __DIR__ . '/../../../v1/adquirente/asaas/asaas_functions.php';







function debug_log($message) {
    $log_file = __DIR__ . '/asaas_cashout_log.txt';
    $timestamp = date('Y-m-d H:i:s');
    
    // Verifica se o message é um array ou objeto e converte para string
    if (is_array($message) || is_object($message)) {
        $message = print_r($message, true); // ou var_export($message, true)
    }
    
    file_put_contents($log_file, "[$timestamp] $message\n", FILE_APPEND);
}




















// Função para validar CPF
function validarCPF($cpf) {
    // Remove caracteres não numéricos
    $cpf = preg_replace('/[^0-9]/', '', $cpf);
    
    // Verifica se tem 11 dígitos
    if (strlen($cpf) != 11) {
        return false;
    }
    
    // Verifica se todos os dígitos são iguais
    if (preg_match('/(\d)\1{10}/', $cpf)) {
        return false;
    }
    
    // Calcula os dígitos verificadores
    for ($t = 9; $t < 11; $t++) {
        for ($d = 0, $c = 0; $c < $t; $c++) {
            $d += $cpf[$c] * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cpf[$c] != $d) {
            return false;
        }
    }
    return true;
}

// Função para validar CNPJ
function validarCNPJ($cnpj) {
    // Remove caracteres não numéricos
    $cnpj = preg_replace('/[^0-9]/', '', $cnpj);
    
    // Verifica se tem 14 dígitos
    if (strlen($cnpj) != 14) {
        return false;
    }
    
    // Verifica se todos os dígitos são iguais
    if (preg_match('/(\d)\1{13}/', $cnpj)) {
        return false;
    }
    
    // Calcula os dígitos verificadores
    for ($i = 0, $j = 5, $soma = 0; $i < 12; $i++) {
        $soma += $cnpj[$i] * $j;
        $j = ($j == 2) ? 9 : $j - 1;
    }
    
    $resto = $soma % 11;
    if ($cnpj[12] != ($resto < 2 ? 0 : 11 - $resto)) {
        return false;
    }
    
    for ($i = 0, $j = 6, $soma = 0; $i < 13; $i++) {
        $soma += $cnpj[$i] * $j;
        $j = ($j == 2) ? 9 : $j - 1;
    }
    
    $resto = $soma % 11;
    return $cnpj[13] == ($resto < 2 ? 0 : 11 - $resto);
}

// Recebe dados do POST (com log do RAW e do erro de JSON, se houver)
$rawInput = file_get_contents('php://input');
debug_log('[IN] RAW: ' . $rawInput);
$data = json_decode($rawInput, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    debug_log('[IN] JSON_ERROR: ' . json_last_error_msg());
}

if (!isset($data['beneficiaryName'], $data['amount'], $data['keypix'])) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Campos obrigatórios ausentes (beneficiaryName, amount, keypix)']);
    exit;
}

$name = $data['beneficiaryName'];
$amount = floatval($data['amount']);
$keypix = $data['keypix'];
$keyType = $data['keyType'] ?? 'EMAIL';
// Normaliza o tipo de chave para o formato esperado pela Asaas
$keyType = strtoupper($keyType);
if ($keyType === 'RANDOM_KEY' || $keyType === 'RANDOM' || $keyType === 'ALEATORIA' || $keyType === 'CHAVE_ALEATORIA') {
    $keyType = 'EVP';
}
debug_log('[PARSED] name=' . $name . ' amount=' . number_format($amount, 2, '.', '') . ' keypix=' . $keypix . ' keyType=' . $keyType);

// Valida CPF/CNPJ apenas se fornecido
if (isset($data['cpf']) && !empty($data['cpf'])) {
    $cpf = preg_replace('/\D+/', '', $data['cpf']);
    $isCPF = strlen($cpf) == 11;
    $isCNPJ = strlen($cpf) == 14;

    if ($isCPF && !validarCPF($cpf)) {
        echo json_encode(['status' => 'error', 'message' => 'CPF inválido']);
        exit;
    }

    if ($isCNPJ && !validarCNPJ($cpf)) {
        echo json_encode(['status' => 'error', 'message' => 'CNPJ inválido']);
        exit;
    }

    if (!$isCPF && !$isCNPJ) {
        echo json_encode(['status' => 'error', 'message' => 'CPF/CNPJ deve ter 11 ou 14 dígitos']);
        exit;
    }
}
 

 



// 2. Criar transferência
// Formatar chave PIX de telefone para formato internacional se necessário
$pixKeyFormatted = $keypix;
if ($keyType === 'PHONE' && substr($keypix, 0, 1) !== '+') {
    // Se for telefone e não tiver +, adicionar +55
    if (substr($keypix, 0, 2) === '55') {
        $pixKeyFormatted = '+' . $keypix;
    } else {
        $pixKeyFormatted = '+55' . $keypix;
    }
}

$payloadAsaas = [
    'value' => $amount,
    'pixAddressKey' => $pixKeyFormatted,
    'pixAddressKeyType' => $keyType,
    'description' => 'Saque de saldo'
];
debug_log('[REQ→ASAAS] transfers ' . json_encode($payloadAsaas));
$transfer = asaasRequest('transfers', 'POST', $payloadAsaas);
debug_log('[RES←ASAAS] ' . json_encode($transfer));



debug_log($transfer) ;



if (empty($transfer['id'])) {
    $errorMessage = 'Erro ao criar transferência';
    if (isset($transfer['errors'])) {
        $errorMessage .= ' - ' . json_encode($transfer['errors']);
    }
    if (isset($transfer['curl_error'])) {
        $errorMessage .= ' - CURL: ' . $transfer['curl_error'];
    }
    debug_log('[ERR] ' . $errorMessage);
    echo json_encode(['status' => 'error', 'message' => $errorMessage, 'asaas' => $transfer]);
    exit;
}

// 3. Retornar resposta
$response = [
    'status' => 200,
    'message' => 'Transferência criada com sucesso',
    'asaas' => $transfer
];
debug_log('[OK] Transfer criada id=' . $transfer['id'] . ' value=' . $transfer['value']);
echo json_encode($response);