<?php
// Carrega configurações primeiro
require_once 'config.php';
require_once 'conectarbanco.php';
require_once '../adquirente/asaas/asaas_functions.php';

// Configurações de erro
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *"); 
header("Access-Control-Allow-Methods: GET, POST, OPTIONS"); 
header("Access-Control-Allow-Headers: Content-Type, Authorization");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200); 
    exit;
}

// CONSULTA DE STATUS (GET)
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    // Verifica se há API key na query string
    if (!isset($_GET['api-key']) || empty($_GET['api-key'])) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Campo obrigatório ausente: api-key"
        ]);
        exit;
    }

    // Verifica se há idTransaction para consulta
    if (!isset($_GET['idTransaction']) || empty($_GET['idTransaction'])) {
        http_response_code(400);
        echo json_encode([
            "status" => "error",
            "message" => "Campo obrigatório ausente: idTransaction"
        ]);
        exit;
    }

    // Conecta ao banco
    $conn = new mysqli('localhost', $config['db_user'], $config['db_pass'], $config['db_name']);

    if ($conn->connect_error) {
        http_response_code(500);
        echo json_encode([
            "status" => "error",
            "message" => "Erro de conexão com banco de dados"
        ]);
        exit;
    }

    // Validar API key
    $sql = "SELECT user_id FROM users_key WHERE api_key = ? AND status = 'ativo'";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $_GET['api-key']);
    $stmt->execute();
    $stmt->store_result();

    if ($stmt->num_rows === 0) {
        http_response_code(401);
        echo json_encode([
            "status" => "error",
            "message" => "API key inválida ou inativa"
        ]);
        $stmt->close();
        $conn->close();
        exit;
    }

    $stmt->bind_result($usuario);
    $stmt->fetch();
    $stmt->close();

    // Consulta o status do pedido
    $sql = "SELECT * FROM solicitacoes WHERE idtransaction = ? AND user_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $_GET['idTransaction'], $usuario);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        http_response_code(404);
        echo json_encode([
            "status" => "error",
            "message" => "Pedido não encontrado"
        ]);
        $stmt->close();
        $conn->close();
        exit;
    }

    $pedido = $result->fetch_assoc();
    $stmt->close();
    $conn->close();

    // Consulta status atual no Asaas
    $statusAsaas = null;
    try {
        $statusAsaas = asaasRequest("payments/{$_GET['idTransaction']}", 'GET');
    } catch (Exception $e) {
        // Se der erro na consulta Asaas, continua com dados do banco
    }

    // Monta resposta
    $response = [
        'status' => 'success',
        'pedido' => [
            'idTransaction' => $pedido['idtransaction'],
            'externalReference' => $pedido['externalreference'],
            'amount' => $pedido['amount'],
            'clientName' => $pedido['client_name'],
            'clientDocument' => $pedido['client_document'],
            'clientEmail' => $pedido['client_email'],
            'realData' => $pedido['real_data'],
            'status' => $pedido['status'],
            'tipoTransacao' => $pedido['tipo_transacao'],
            'descTransacao' => $pedido['desc_transacao'],
            'depositoLiquido' => $pedido['deposito_liquido'],
            'adquirenteRef' => $pedido['adquirente_ref']
        ]
    ];

    // Adiciona dados do Asaas se disponível
    if ($statusAsaas && isset($statusAsaas['status'])) {
        $response['asaas'] = [
            'status' => $statusAsaas['status'],
            'dueDate' => $statusAsaas['dueDate'] ?? null,
            'paymentDate' => $statusAsaas['paymentDate'] ?? null,
            'value' => $statusAsaas['value'] ?? null
        ];

        // Atualiza status no banco se for diferente
        if ($statusAsaas['status'] !== $pedido['status']) {
            $conn = new mysqli('localhost', $config['db_user'], $config['db_pass'], $config['db_name']);
            if (!$conn->connect_error) {
                $updateSql = "UPDATE solicitacoes SET status = ? WHERE idtransaction = ?";
                $updateStmt = $conn->prepare($updateSql);
                $updateStmt->bind_param("ss", $statusAsaas['status'], $_GET['idTransaction']);
                $updateStmt->execute();
                $updateStmt->close();
                $conn->close();
                
                $response['pedido']['status'] = $statusAsaas['status'];
                $response['statusAtualizado'] = true;
            }
        }
    }

    echo json_encode($response);
    exit;
}

// PROCESSAMENTO DE PAGAMENTO (POST)
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        "status" => "error", 
        "message" => "Método não permitido. Use POST para pagamentos ou GET para consultas."
    ]);
    exit;
}

// Lê JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

// Segurança básica contra JSON inválido
if (!is_array($data)) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'JSON inválido.']);
    exit;
}

// Validação da API key
if (!isset($data['api-key']) || empty($data['api-key'])) {
    http_response_code(400); 
    echo json_encode([
        "status" => "error", 
        "message" => "Campo obrigatório ausente: api-key"
    ]);
    exit;
}

// Conecta ao banco
$conn = new mysqli('localhost', $config['db_user'], $config['db_pass'], $config['db_name']);

if ($conn->connect_error) {
    http_response_code(500);
    echo json_encode([
        "status" => "error", 
        "message" => "Erro de conexão com banco de dados"
    ]);
    exit;
}

// Validar API key
$sql = "SELECT user_id FROM users_key WHERE api_key = ? AND status = 'ativo'";
$stmt = $conn->prepare($sql);
$stmt->bind_param("s", $data['api-key']);
$stmt->execute();
$stmt->store_result();

if ($stmt->num_rows === 0) {
    http_response_code(401);
    echo json_encode([
        "status" => "error", 
        "message" => "API key inválida ou inativa"
    ]);
    $stmt->close();
    $conn->close();
    exit;
}

$stmt->bind_result($usuario);
$stmt->fetch();
$stmt->close();

// Valida campos mínimos
$required = ['name', 'cpf', 'amount', 'tipoPagamento', 'email'];
foreach ($required as $r) {
    if (!isset($data[$r]) || $data[$r] === '') {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => "Campo obrigatório ausente: {$r}"]);
        $conn->close();
        exit;
    }
}

// Sanitiza CPF/CNPJ (remove máscara)
$cpfCnpj = preg_replace('/\D+/', '', $data['cpf']);

// Valor numérico (troca vírgula por ponto caso venha em formato brasileiro)
$valor = str_replace(',', '.', $data['amount']);
$valor = (float)$valor;

// Verifica se é parcelamento
$installmentCount = isset($data['installmentCount']) ? (int)$data['installmentCount'] : 1;
$isInstallment = $installmentCount > 1;

// Cria cliente no Asaas
$customer = asaasRequest('customers', 'POST', [
    'name'    => $data['name'],
    'cpfCnpj' => $cpfCnpj,
    'email'   => $data['email'],
]);

if (empty($customer['id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Erro ao criar cliente', 'asaas' => $customer]);
    $conn->close();
    exit;
}

// Billing type
$tipo = strtoupper($data['tipoPagamento']); // PIX, BOLETO, CREDIT_CARD
$protocolo = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
$url_base = $protocolo . "://" . $_SERVER['HTTP_HOST'] . "/api/v1/webhook/asaas.php";

$payload = [
    'customer'   => $customer['id'],
    'billingType'=> $tipo,
    'dueDate'    => date('Y-m-d'),
    "callback"   => [
        "cancelUrl"   => $url_base,
        "expiredUrl"  => $url_base,
        "successUrl"  => $url_base
    ]
];

// Configura valor ou parcelamento
if ($isInstallment) {
    // Para parcelamento, usa installmentValue e installmentCount
    $installmentValue = $valor / $installmentCount;
    $payload['installmentCount'] = $installmentCount;
    $payload['installmentValue'] = round($installmentValue, 2);
} else {
    // Para pagamento à vista, usa value
    $payload['value'] = $valor;
}

// Se cartão, anexar dados
if ($tipo === 'CREDIT_CARD') {
    if (empty($data['creditCard']) || empty($data['creditCardHolderInfo'])) {
        echo json_encode(['status' => 'error', 'message' => 'Dados do cartão ausentes']);
        $conn->close();
        exit;
    }

    // Garante que holderName está presente
    if (empty($data['creditCard']['holderName'])) {
        $data['creditCard']['holderName'] = $data['creditCardHolderInfo']['name'] ?? $data['name'];
    }

    $payload['creditCard']           = $data['creditCard'];
    $payload['creditCardHolderInfo'] = $data['creditCardHolderInfo'];
}

// Cria cobrança no Asaas
$payment = asaasRequest('payments', 'POST', $payload);

if (empty($payment['id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Erro ao criar cobrança', 'asaas' => $payment]);
    $conn->close();
    exit;
}

// Registra no banco de dados
$user_id = $data['user_id'] ?? $usuario;
$externalreference = bin2hex(random_bytes(16));
$amount = $valor;
$client_name = $data['name'];
$client_document = $cpfCnpj;
$client_email = $data['email'];
$real_data = date('Y-m-d');
$status = 'WAITING_FOR_APPROVAL';
$qrcode_pix = $tipo === 'PIX' ? ($payment['pix']['payload'] ?? '') : '';
$paymentcode = $tipo === 'PIX' ? ($payment['pix']['payload'] ?? '') : '';
$idtransaction = $payment['id'];
$paymentCodeBase64 = $tipo === 'PIX' && isset($payment['pix']['payload']) ? base64_encode($payment['pix']['payload']) : null;
$adquirente_ref = 'asaas';
$taxa_cash_in = 0.00;
$tipo_transacao = $tipo;

// Obtém o nome do produto do payload
$productName = $data['productName'] ?? '';

// Ajusta descrição para incluir nome do produto e parcelamento
if ($isInstallment) {
    $baseDesc = $tipo === 'PIX' ? "Pagamento via Pix em {$installmentCount}x" : 
               ($tipo === 'CREDIT_CARD' ? "Pagamento via Cartão em {$installmentCount}x" : '');
    $desc_transacao = $productName ? "{$productName} - {$baseDesc}" : $baseDesc;
} else {
    $baseDesc = $tipo === 'PIX' ? 'Pagamento via Pix' : 
               ($tipo === 'CREDIT_CARD' ? 'Pagamento via Cartão' : '');
    $desc_transacao = $productName ? "{$productName} - {$baseDesc}" : $baseDesc;
}

$deposito_liquido = $amount;
$taxa_pix_cash_in_adquirente = 0.00;
$taxa_pix_cash_in_valor_fixo = 0.00;
$client_telefone = $data['telefone'] ?? null;
$executor_ordem = null;
$descricao_transacao = $desc_transacao;
$postback = null;

$stmt = $conn->prepare("INSERT INTO solicitacoes (user_id, externalreference, amount, client_name, client_document, client_email, real_data, status, qrcode_pix, paymentcode, idtransaction, paymentCodeBase64, adquirente_ref, taxa_cash_in, tipo_transacao, desc_transacao, deposito_liquido, taxa_pix_cash_in_adquirente, taxa_pix_cash_in_valor_fixo, client_telefone, executor_ordem, descricao_transacao, postback) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

if ($stmt) {
    $stmt->bind_param(
        "ssdssssssssssdsssssssss",
        $user_id, $externalreference, $amount, $client_name, $client_document, 
        $client_email, $real_data, $status, $qrcode_pix, $paymentcode, 
        $idtransaction, $paymentCodeBase64, $adquirente_ref, $taxa_cash_in, 
        $tipo_transacao, $desc_transacao, $deposito_liquido, $taxa_pix_cash_in_adquirente, 
        $taxa_pix_cash_in_valor_fixo, $client_telefone, $executor_ordem, 
        $descricao_transacao, $postback
    );
    $stmt->execute();
    $stmt->close();
}

$conn->close();

// Monta resposta
$response = [
    'status'        => 'success',
    'idTransaction' => $payment['id'],
    'tipo'          => $tipo,
    'asaas'         => $payment
];

// Adiciona informações de parcelamento se aplicável
if ($isInstallment) {
    $response['installmentCount'] = $installmentCount;
    $response['installmentValue'] = round($valor / $installmentCount, 2);
    $response['totalValue'] = $valor;
}

// Dados extras conforme tipo
if ($tipo === 'PIX') {
    $response['paymentCode'] = $payment['pix']['payload'] ?? null;
    $response['qrCodeImage'] = $payment['pix']['encodedImage'] ?? null;

    // Se não vier, chama o endpoint pixQrCode
    if (empty($response['qrCodeImage'])) {
        $pixResp = asaasRequest("payments/{$payment['id']}/pixQrCode", 'GET');
        $response['paymentCode'] = $response['paymentCode'] ?: ($pixResp['payload'] ?? null);
        $response['qrCodeImage'] = $response['qrCodeImage'] ?: ($pixResp['encodedImage'] ?? null);
        $response['asaas_pix']   = $pixResp;
    }
}

if ($tipo === 'BOLETO') {
    $response['boletoUrl'] = $payment['bankSlipUrl'] ?? null;
}

echo json_encode($response);
?>
