<?php
/**
 * Plugin Name: WooCommerce PIX Gateway
 * Plugin URI: https://velozpay.online
 * Description: Gateway de pagamento PIX para WooCommerce integrado com Asaas
 * Version: 1.0.0
 * Author: Seu Nome
 * Author URI: https://seusite.com
 * Text Domain: woocommerce-pix-gateway
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * WC requires at least: 5.0
 * WC tested up to: 8.0
 * Requires PHP: 7.4
 */

// Prevenir acesso direto
if (!defined('ABSPATH')) {
    exit;
}

// Verificar se WooCommerce está ativo
if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    return;
}

// Definir constantes
define('WC_PIX_GATEWAY_VERSION', '1.0.0');
define('WC_PIX_GATEWAY_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WC_PIX_GATEWAY_PLUGIN_PATH', plugin_dir_path(__FILE__));

/**
 * Classe principal do plugin
 */
class WC_PIX_Gateway_Plugin {
    
    public function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        add_action('init', array($this, 'load_textdomain'));
    }
    
    /**
     * Inicializar o plugin
     */
    public function init() {
        // Adicionar gateway ao WooCommerce
        add_filter('woocommerce_payment_gateways', array($this, 'add_gateway'));
        
        // Carregar classe do gateway
        require_once WC_PIX_GATEWAY_PLUGIN_PATH . 'includes/class-wc-pix-gateway.php';
        
        // Adicionar scripts e estilos
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
        
        // Adicionar menu de configurações
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Adicionar links de ação no plugin
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'plugin_action_links'));
    }
    
    /**
     * Carregar traduções
     */
    public function load_textdomain() {
        load_plugin_textdomain('woocommerce-pix-gateway', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    /**
     * Adicionar gateway ao WooCommerce
     */
    public function add_gateway($gateways) {
        $gateways[] = 'WC_PIX_Gateway';
        return $gateways;
    }
    
    /**
     * Carregar scripts e estilos no frontend
     */
    public function enqueue_scripts() {
        if (is_checkout()) {
            wp_enqueue_script('wc-pix-gateway', WC_PIX_GATEWAY_PLUGIN_URL . 'assets/js/frontend.js', array('jquery'), WC_PIX_GATEWAY_VERSION, true);
            wp_enqueue_style('wc-pix-gateway', WC_PIX_GATEWAY_PLUGIN_URL . 'assets/css/frontend.css', array(), WC_PIX_GATEWAY_VERSION);
            
            wp_localize_script('wc-pix-gateway', 'wc_pix_gateway', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('wc_pix_gateway_nonce'),
                'strings' => array(
                    'loading' => __('Carregando...', 'woocommerce-pix-gateway'),
                    'error' => __('Erro ao processar pagamento', 'woocommerce-pix-gateway'),
                    'success' => __('Pagamento processado com sucesso!', 'woocommerce-pix-gateway')
                )
            ));
        }
    }
    
    /**
     * Carregar scripts e estilos no admin
     */
    public function admin_enqueue_scripts($hook) {
        if ('woocommerce_page_wc-settings' === $hook && isset($_GET['section']) && $_GET['section'] === 'pix_gateway') {
            wp_enqueue_script('wc-pix-gateway-admin', WC_PIX_GATEWAY_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), WC_PIX_GATEWAY_VERSION, true);
            wp_enqueue_style('wc-pix-gateway-admin', WC_PIX_GATEWAY_PLUGIN_URL . 'assets/css/admin.css', array(), WC_PIX_GATEWAY_VERSION);
        }
    }
    
    /**
     * Adicionar menu de configurações
     */
    public function add_admin_menu() {
        // Adicionar submenu no WooCommerce
        add_submenu_page(
            'woocommerce',
            'Configurações PIX Gateway',
            'PIX Gateway',
            'manage_woocommerce',
            'wc-pix-gateway-settings',
            array($this, 'admin_page')
        );
    }
    
    /**
     * Página de configurações do admin
     */
    public function admin_page() {
        // Verificar permissões
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Você não tem permissão para acessar esta página.');
        }
        
        // Incluir a página administrativa
        include WC_PIX_GATEWAY_PLUGIN_PATH . 'admin/admin-page.php';
    }
    
    /**
     * Links de ação do plugin
     */
    public function plugin_action_links($links) {
        $plugin_links = array(
            '<a href="' . admin_url('admin.php?page=wc-settings&tab=checkout&section=pix_gateway') . '">' . __('Configurações', 'woocommerce-pix-gateway') . '</a>',
        );
        return array_merge($plugin_links, $links);
    }
}

// Inicializar o plugin
new WC_PIX_Gateway_Plugin();

// Ativação do plugin
register_activation_hook(__FILE__, 'wc_pix_gateway_activate');
function wc_pix_gateway_activate() {
    // Verificar se WooCommerce está ativo
    if (!class_exists('WooCommerce')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(__('Este plugin requer o WooCommerce ativo.', 'woocommerce-pix-gateway'));
    }
    
    // Adicionar opções padrão
    add_option('wc_pix_gateway_api_url', 'https://velozpay.online/api/v1/gateway/');
    add_option('wc_pix_gateway_sandbox', 'yes');
    add_option('wc_pix_gateway_debug', 'no');
    add_option('wc_pix_gateway_expiration_time', 30);
}

// Desativação do plugin
register_deactivation_hook(__FILE__, 'wc_pix_gateway_deactivate');
function wc_pix_gateway_deactivate() {
    // Limpar opções se necessário
}
